<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\Product;
use App\Models\Customer;
use Illuminate\Support\Facades\Mail;
use App\Mail\IngredientLowStockAlert;
use Illuminate\Foundation\Testing\RefreshDatabase;

class OrderTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        $this->seed();
        Mail::fake();
    }

    public function test_order_creation(): void
    {
        $customer = Customer::first();
        $product = Product::first();

        $response = $this->postJson('/api/orders', [
            'customer_id' => $customer->id,
            'products' => [
                ['product_id' => $product->id, 'quantity' => 2]
            ]
        ]);

        $response->assertStatus(201)
            ->assertJsonStructure([
                'message',
                'order' => [
                    'id',
                    'customer_id',
                    'total_price',
                    'items' => [
                        '*' => ['id', 'product_id', 'quantity', 'unit_price']
                    ]
                ]
            ]);

        $this->assertDatabaseHas('orders', [
            'customer_id' => $customer->id,
            'total_price' => $product->price * 2
        ]);
    }

    public function test_low_stock_notification(): void
    {
        $customer = Customer::first();
        $product = Product::with('ingredients')->first();
        $ingredient = $product->ingredients->first();

        // Set stock just above threshold
        $ingredient->update([
            'current_stock' => ($ingredient->initial_stock * 0.5) + 1,
            'low_stock_notification_sent' => false
        ]);

        $this->postJson('/api/orders', [
            'customer_id' => $customer->id,
            'products' => [
                ['product_id' => $product->id, 'quantity' => 1]
            ]
        ]);

        Mail::assertSent(IngredientLowStockAlert::class);
    }
}
