<?php

namespace App\Trait\Logic;

use App\Jobs\HandleLogicTaskJob;
use App\Models\LogicOperation;
use App\Models\LogicTask;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

trait CheckOperationObserver
{
    /**
     * @return void
     */
    public static function bootCheckOperationObserver(): void
    {
        static::created(static function (Model $model) {
            self::checkAndSetLogicId($model, 'create');
        });

        static::updated(static function (Model $model) {
            self::checkAndSetLogicId($model, 'update');
        });

        static::deleted(static function (Model $model) {
            self::checkAndSetLogicId($model, 'delete');
        });
    }

    /**
     * Check if the given operation exists in the logic_operations table
     * and set the logic_id attribute in the model accordingly.
     *
     * @param Model $model
     * @param string $operationType
     * @return void
     */
    protected static function checkAndSetLogicId(Model $model, string $operationType): void
    {
        $modelClass = class_basename($model);
        $operation = $operationType . '-' . Str::snake($modelClass,'-');

        $operationRecords = LogicOperation::where('operation', $operation)->get();

        foreach ($operationRecords as $operationRecord) {
            self::assignLogicToUser($model, $operationRecord->logic_id);
        }
    }

    /**
     * Assign logic tasks to the model based on the logic_id.
     *
     * @param Model $model
     * @param int $logicId
     * @return void
     */
    public static function assignLogicToUser(Model $model, int $logicId): void
    {
        $logicTasks = LogicTask::where('logic_id', $logicId)->whereNull('parent_id')->get();
        $logicModel = $model->logics()->create(['logic_id' => $logicId]);

        foreach ($logicTasks as $task) {
            dispatch(new HandleLogicTaskJob($task, $logicModel, $model));
        }
    }
}
