<?php
namespace App\Services\Auth;

use App\Models\Client;
use App\Models\Employee;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Cache;
use App\Exceptions\InvalidOtpException;
use Illuminate\Database\Eloquent\Model;
use App\Notifications\SendOtpNotification;
use Illuminate\Support\Facades\Notification;
use App\Exceptions\InvalidEmailAndPasswordCombinationException;

class AuthService
{
    private string $model;
    private string $guard;

    /**
     * @throws InvalidEmailAndPasswordCombinationException
     */
    public function attempt(Request $request): array
    {
        $query = $this->getModel()->query();
        $user = match ($this->getGuard()) {
            'customer' => $this->findCustomer($query, $request),
            'employee' => $this->findEmployee($query, $request),
            default => throw new InvalidEmailAndPasswordCombinationException(),
        };
        $this->setLastLogin($user);
        return [
            'user' => $user,
            'token' => $user->createToken($this->getGuard())->plainTextToken,
        ];
    }

    /**
     * Find a client by email or phone.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Database\Eloquent\Model
     * @throws \App\Exceptions\InvalidEmailAndPasswordCombinationException
     */
    protected function findCustomer($query, Request $request): Model
    {
        $user = $query->where('email', $request->email)->orWhere('phone', $request->phone)
            ->first();

        if (!$user) {
            throw new InvalidEmailAndPasswordCombinationException();
        }

        return $user;
    }

    /**
     * Find an employee by email.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Database\Eloquent\Model
     * @throws \App\Exceptions\InvalidEmailAndPasswordCombinationException
     */
    protected function findEmployee($query, Request $request): Model
    {
        $user = $query->where('email', $request->email)->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            throw new InvalidEmailAndPasswordCombinationException();
        }

        return $user;
    }



    /**
     * Register a new user and return the user and its token.
     *
     * @param Request $request
     * @return array
     */
    public function register(Request $request): array
    {
        $user = $this->getModel()->create([
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'password' => $request->password,
            'address' => $request->address,
            'lat' => $request->lat,
            'lng' => $request->lng,
            'country_id' => $request->country_id,
            'city_id' => $request->city_id,
            'district_id' => $request->district_id,
        ]);

        $this->setLastLogin($user);

        return [
            'user' => $user,
            'token' => $user->createToken($this->getGuard())->plainTextToken,
        ];
    }
    /**
     * @param  String $model
     * @return $this
     */
    public function setModel(string $model): self
    {
        $this->model = $model;
        return $this;
    }

    /**
     * @return Model
     */
    public function getModel(): Model
    {
        return new $this->model();
    }

    /**
     * @param  String $guard
     * @return $this
     */
    public function setGuard(string $guard): self
    {
        $this->guard = $guard;
        return $this;
    }

    /**
     * @return String
     */
    public function getGuard(): string
    {
        return $this->guard;
    }

    /**
     * @param  $user
     * @return bool
     */
    public function setLastLogin($user): bool
    {
        return $user->update(['last_login' => now()]);
    }
}
