<?php

namespace App\Http\Resources\API\Order;

use Illuminate\Http\Request;
use App\Http\Resources\API\ClientResource;
use App\Http\Resources\API\EmployeeResource;
use App\Http\Resources\ClientAddressResource;
use Illuminate\Http\Resources\Json\JsonResource;

class OrderResource extends JsonResource
{
    /**
     * Transform the resource into an array.
     *
     * @return array<string, mixed>
     */
    public function toArray(Request $request): array
    {
        return [
            'id' => $this->id,
            'order_number' => $this->order_number,
            'delegate' => $this->whenLoaded('delegate', fn() => new EmployeeResource($this->delegate)),
            'client' => $this->whenLoaded('client', fn() => new ClientResource($this->client)),
            'subtotal' => $this->subtotal,
            'total' => $this->total,
            'shipping_cost' => $this->shipping_cost,
            'order_discount' => $this->order_discount ?? 0,
            'product_discount' => $this->calculateProductDiscount(),
            'items' => OrderItemResource::collection($this->whenLoaded('items')),
            'note_history' => OrderNoteHistoryResource::collection($this->whenLoaded('noteHistory')),
            'status_history' => $this->whenLoaded('statusHistory', fn() => new OrderStatusHistoryResource($this->statusHistory)),
            'payment' => new PaymentResource($this->payment),
            'address' => $this->whenLoaded('clientAddress', fn() => new ClientAddressResource($this->clientAddress)),

            'quantity' => $this->items->sum('quantity'),
            'order_status' => $this->order_status,
            'is_dashboard' => $this->is_dashboard,
            'notes' => $this->notes,
            // 'discounts' => DiscountResource::collection($this->discounts),
            'created_at' => $this->created_at,
            'updated_at' => $this->updated_at,
        ];
    }

    private function calculateProductDiscount(): float
    {
        return $this->items->sum(function ($item) {
            $price = $item->price;
            $quantity = $item->quantity;
            $discount = 0;

            if ($item->discount_type === 'percentage') {
                $discount = ($price * $item->discount / 100) * $quantity;
            } elseif ($item->discount_type === 'fixed') {
                $discount = ($item->discount * $quantity);
            }

            return $discount;
        });
    }
}
