<?php

namespace App\Http\Controllers\API\Category;

use App\Filters\Global\NameFilter;
use App\Filters\Global\OrderByFilter;
use App\Http\Controllers\Controller;
use App\Http\Requests\API\Category\CategoryRequest;
use App\Http\Requests\Global\DeleteAllRequest;
use App\Http\Requests\Global\PageRequest;
use App\Http\Resources\API\Category\CategoryResource;
use App\Models\Category;
use App\Services\Global\UploadService;
use Illuminate\Http\JsonResponse;
use Illuminate\Pipeline\Pipeline;

class CategoryController extends Controller
{
    /**
     * @param PageRequest $request
     * @return JsonResponse
     */
    public function index(PageRequest $request): JsonResponse
    {
        $query = app(Pipeline::class)
            ->send(Category::with('products'))
            ->through([NameFilter::class, OrderByFilter::class])
            ->thenReturn();

        return successResponse(fetchData($query, $request->pageSize, CategoryResource::class));
    }

    /**
     * @param CategoryRequest $request
     * @return JsonResponse
     */
    public function store(CategoryRequest $request): JsonResponse
    {
        $category = Category::create($request->validated());

        return successResponse(new CategoryResource($category), __('api.created_success'));
    }

    /**
     * @param Category $category
     * @return JsonResponse
     */
    public function show(Category $category): JsonResponse
    {
        return successResponse(new CategoryResource($category->load(['products'])));
    }

    /**
     * @param CategoryRequest $request
     * @param Category $category
     * @return JsonResponse
     */
    public function update(CategoryRequest $request, Category $category): JsonResponse
    {
        $category->update($request->validated());

        return successResponse(new CategoryResource($category), __('api.updated_success'));
    }

    /**
     * @param Category $category
     * @return JsonResponse
     */
    public function destroy(Category $category): JsonResponse
    {
        UploadService::delete($category->image);
        $category->delete();
        return successResponse(msg: __('api.deleted_success'));
    }

    /**
     * @param DeleteAllRequest $request
     * @return JsonResponse
     */
    public function destroyAll(DeleteAllRequest $request): JsonResponse
    {
        Category::whereIn('id', $request->ids)->delete();
        return successResponse(msg: __('api.deleted_success'));
    }

    /**
     * @param DeleteAllRequest $request
     * @return JsonResponse
     */
    public function restoreAll(DeleteAllRequest $request): JsonResponse
    {
        Category::whereIn('id', $request->ids)->onlyTrashed()->restore();

        return successResponse(msg: __('api.restored_success'));
    }

    /**
     * @param DeleteAllRequest $request
     * @return JsonResponse
     */
    public function forceDeleteAll(DeleteAllRequest $request): JsonResponse
    {
        Category::whereIn('id', $request->ids)->onlyTrashed()->forceDelete();

        return successResponse(msg: __('api.deleted_success'));
    }

    /**
     * @param int $id
     * @return JsonResponse
     */
    public function forceDelete(int $id): JsonResponse
    {
        Category::onlyTrashed()->findOrFail($id)->forceDelete();
        return successResponse(msg: __('api.deleted_success'));
    }

    /**
     * @param int $id
     * @return JsonResponse
     */
    public function restore(int $id): JsonResponse
    {
        Category::onlyTrashed()->findOrFail($id)?->restore();

        return successResponse(msg: __('api.restored_success'));
    }

}
